/*
 * CERApplet.java
 *
 * Created on 2006-01-18, 11:20
 */

package ch.hslu.cm.cer;

import ch.hslu.cm.cer.diagram.*;
import ch.hslu.cm.cer.model.CEREntitySet;
import ch.hslu.cm.cer.model.ConceptualERModel;
import ch.hslu.cm.cer.model.CERRelationshipSet;
import ch.randelshofer.util.*;
import ch.randelshofer.swingcss.*;
import java.awt.*;
import java.awt.geom.*;
import java.awt.event.*;
import java.net.*;
import java.io.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.EmptyBorder;
import javax.swing.border.LineBorder;
import javax.swing.text.html.*;
import org.jhotdraw.app.action.edit.DeleteAction;
import org.jhotdraw.app.action.edit.DuplicateAction;
import org.jhotdraw.app.action.edit.SelectAllAction;
import org.jhotdraw.draw.AttributeKey;
import org.jhotdraw.draw.AttributeKeys;
import org.jhotdraw.draw.DefaultDrawingEditor;
import org.jhotdraw.draw.Drawing;
import org.jhotdraw.draw.DrawingEditor;
import org.jhotdraw.draw.DrawingView;
import org.jhotdraw.draw.TextFigure;
import org.jhotdraw.draw.event.ToolAdapter;
import org.jhotdraw.draw.event.ToolEvent;
import org.jhotdraw.draw.tool.ConnectionTool;
import org.jhotdraw.draw.tool.CreationTool;
import org.jhotdraw.draw.tool.DelegationSelectionTool;
import org.jhotdraw.draw.tool.Tool;
import org.jhotdraw.gui.Worker;
import org.jhotdraw.undo.UndoRedoManager;
import org.jhotdraw.util.ResourceBundleUtil;
import org.jhotdraw.xml.*;

/**
 * Applet with an editor for General Entity-Relationship diagrams.
 *
 * @author  wrandels
 */
public class CERApplet extends JApplet {
    private final static String VERSION = "0.1";
    private final static String NAME = "ConceptModeler[Conceptual E-R]";
    
    private LinkedList<Action> drawingActions;
    private LinkedList<Action> selectionActions;
    private UndoRedoManager undoManager;
    protected ConceptualERDiagram problemDiagram, solutionDiagram;
    protected ResourceBundleUtil labels;
    private DrawingEditor editor;
    protected StyleManager stylemanager;
    protected StyleSheet stylesheet;
    
    private class ToolButtonListener implements ItemListener {
        private Tool tool;
        public ToolButtonListener(Tool t) {
            this.tool = t;
        }
        @Override
        public void itemStateChanged(ItemEvent evt) {
            if (evt.getStateChange() == ItemEvent.SELECTED) {
                editor.setTool(tool);
            }
        }
    }
    
    protected DrawingView getDrawingView() {
        return drawingView;
    }
    
    /**
     * Handles toggle buttons for the tools.
     */
    private class ToolHandler extends ToolAdapter {
        private JToggleButton defaultToolButton;
        public ToolHandler(JToggleButton defaultToolButton) {
            this.defaultToolButton = defaultToolButton;
        }
        
        @Override
        public void toolDone(ToolEvent event) {
            defaultToolButton.setSelected(true);
        }
        
    }
    
    @Override
    public String getParameter(String name) {
        try {
            return super.getParameter(name);
        } catch (NullPointerException e) {
            return null;
        }
    }
    
    /**
     * Initializes the applet CERApplet
     */
    @Override
    public void init() {
        ResourceBundleUtil.setVerbose(true);

        if (getParameter("locale") != null) {
            try  {
                Locale locale = new Locale(getParameter("locale"));
                LocaleUtil.setDefault(locale);
                Locale.setDefault(locale);
            } catch (SecurityException e) {
                // There is nothing we can do about this.
            }
        }
        stylemanager = new StyleManager();
        if (getParameter("stylesheet") != null) {
            BufferedReader in = null;
            StringBuilder stylesheetData = new StringBuilder();
            try  {
            URL stylesheetURL = new URL(getDocumentBase(), getParameter("stylesheet"));
                in = new BufferedReader(new InputStreamReader(stylesheetURL.openStream(),"UTF8"));
                for (String line = in.readLine(); line != null; line = in.readLine()) {
                    stylesheetData.append(line);
                    stylesheetData.append('\n');
                }
                stylesheet = new StyleSheet();
                stylesheet.loadRules(new StringReader(stylesheetData.toString()), stylesheetURL);
                CSSLoader loader = new CSSLoader();
                loader.load(new StringReader(stylesheetData.toString()), stylemanager);
            } catch (IOException e) {
                e.printStackTrace();
            } finally {
                try { if (in != null) in.close(); } catch (IOException e) {}
            }
        }
        
        labels = ResourceBundleUtil.getBundle("ch.hslu.cm.cer.Labels");
        
        try {
            UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
        } catch (Throwable e) {
            // Do nothing.
            // If we can't set the desired look and feel, UIManager does
            // automaticaly the right thing for us.
        }
        Container c = getContentPane();
        c.setLayout(new BoxLayout(c, BoxLayout.Y_AXIS));
        for (StringTokenizer tt = new StringTokenizer(getAppletInfo(), "\n"); tt.hasMoreTokens(); ) {
            c.add(new JLabel(tt.nextToken()));
        }
        reset();
    }
    
    /**
     * Does additional initialisation steps.
     * This is for use by subclasses.
     */
    protected void initAdditional() {
        
    }
    
    /**
     * Resets the diagram to its initial state.
     */
    public void reset() {
        new Worker() {
            @Override
            public Object construct() {
                Object result = null;
                
                editor = new DefaultDrawingEditor();
                InputStream in = null;
                
                try {
                    NanoXMLDOMInput domi = null;
                    if (getParameter("data") != null) {
                        domi = new NanoXMLDOMInput(new CERFactory(), new StringReader(getParameter("data")));
                    } else if (getParameter("datafile") != null) {
                        URL url = new URL(getDocumentBase(), getParameter("datafile"));
                        in = url.openConnection().getInputStream();
                        domi = new NanoXMLDOMInput(new CERFactory(), in);
                    }
                    if (domi != null) {
                        domi.openElement("ConceptModelerCER");
                        
                        String instructionsText;
                        if (domi.getElementCount("Instructions") > 0) {
                            domi.openElement("Instructions");
                            instructionsText = (String) domi.readObject();
                            domi.closeElement();
                        } else {
                            instructionsText = "";
                        }
                        ConceptualERDiagram pDiagram;
                        if (domi.getElementCount("Problem") > 0) {
                            domi.openElement("Problem");
                            ConceptualERModel pSimulation = (ConceptualERModel) domi.readObject(0);
                            pDiagram = (ConceptualERDiagram) domi.readObject(1);
                            domi.closeElement();
                        } else {
                            pDiagram = new ConceptualERDiagram();
                        }
                        ConceptualERDiagram sDiagram;
                        if (domi.getElementCount("Solution") > 0) {
                            domi.openElement("Solution");
                            ConceptualERModel sSimulation = (ConceptualERModel) domi.readObject(0);
                            sDiagram = (ConceptualERDiagram) domi.readObject(1);
                            domi.closeElement();
                            domi.closeElement();
                        } else {
                            sDiagram = new ConceptualERDiagram();
                        }
                        result = new Object[] {
                            instructionsText,
                            pDiagram,
                            sDiagram
                        };
                    }
                } catch (Throwable t) {
                    result = t;
                } finally {
                    try { if (in != null) in.close(); } catch (IOException e) {}
                }
                return result;
            }
            @Override
            public void finished() {
                Container c = getContentPane();
                c.setLayout(new BorderLayout());
                c.removeAll();
                
                Object result = getValue();
                initComponents();
                if (result != null && (result instanceof Object[])) {
                    problemDiagram = (ConceptualERDiagram) ((Object[]) result)[1];
                    solutionDiagram = (ConceptualERDiagram) ((Object[]) result)[2];
                } else {
                    problemDiagram = new ConceptualERDiagram();
                    solutionDiagram = new ConceptualERDiagram();
                }
                /*
                problemDiagram.setLocale(getLocale());
                solutionDiagram.setLocale(getLocale());
                */
                drawingView.setDrawing(problemDiagram);
                drawingView.setEmptyDrawingMessage(labels.getString("emptySolutionMessage"));
                editor.add(drawingView);
                undoManager = new UndoRedoManager();
                undoManager.setLocale(getLocale());
                problemDiagram.addUndoableEditListener(undoManager);
                //initTools();
                addDrawActionsTo(toolBar);
                
                if (result != null && (result instanceof Throwable)) {
                    problemDiagram.add(new TextFigure(result.toString()));
                    ((Throwable) result).printStackTrace();
                }
                
                scrollPane.setBorder(new EmptyBorder(0,0,0,0));
               ((JComponent) getContentPane()).setBorder(new LineBorder(Color.gray));
                
                initAdditional();
                
                //
                ((JComponent) CERApplet.this.getContentPane()).putClientProperty("css.class","applet");
               drawingView.getComponent().putClientProperty("css.class","drawing");
                toolBar.putClientProperty("css.class","toolbar");
                toolBar.setOpaque(true);
                stylemanager.style(CERApplet.this);

                // 
                c.validate();
            }}.start();
        }
    public void addDrawActionsTo(JToolBar tb) {
        ResourceBundleUtil cerLabels = ResourceBundleUtil.getBundle("ch.hslu.cm.cer.Labels", getLocale());
        ResourceBundleUtil drawLabels = ResourceBundleUtil.getBundle("org.jhotdraw.draw.Labels", getLocale());
        
        JToggleButton t;
        Tool tool;
        ButtonGroup group;
        HashMap<AttributeKey,Object> attributes;
        CERConnectionFigure rf;
        
        group = new ButtonGroup();
        // Selection tool
        Tool selectionTool = new DelegationSelectionTool(getDrawingActions(), getSelectionActions());
        editor.setTool(selectionTool);
        t = new JToggleButton();
        ToolHandler toolHandler = new ToolHandler(t);
        drawLabels.configureToolBarButton(t, "selectionTool");
        t.setSelected(true);
        t.addItemListener(
                new ToolButtonListener(selectionTool)
                );
        t.setFocusable(false);
        group.add(t);
        tb.add(t);
        tb.addSeparator();
        
        // AttributeKeys for the entities
        attributes = new HashMap<AttributeKey,Object>();
        attributes.put(AttributeKeys.FILL_COLOR, ConceptualERDiagram.ATTRIBUTE_FILL_COLOR);
        attributes.put(AttributeKeys.STROKE_COLOR, ConceptualERDiagram.ATTRIBUTE_STROKE_COLOR);
        attributes.put(AttributeKeys.TEXT_COLOR, ConceptualERDiagram.ATTRIBUTE_TEXT_COLOR);
        
        // Strong CEREntitySet
        t = new JToggleButton();
        cerLabels.configureToolBarButton(t, "edit.createEntitySetStrong");
        tool = new CreationTool(new CEREntitySetFigure(CEREntitySet.TYPE_STRONG), attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
        
        // Weak CEREntitySet
        t = new JToggleButton();
        cerLabels.configureToolBarButton(t, "edit.createEntitySetWeak");
        tool = new CreationTool(new CEREntitySetFigure(CEREntitySet.TYPE_WEAK), attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
        
        // Strong CERRelationshipSet
        t = new JToggleButton();
        cerLabels.configureToolBarButton(t, "edit.createRelationshipSetStrong");
        tool = new CreationTool(new CERRelationshipSetFigure(CERRelationshipSet.TYPE_STRONG), attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
        
        // Identifying CERRelationshipSet
        t = new JToggleButton();
        cerLabels.configureToolBarButton(t, "edit.createRelationshipSetIdentifying");
        tool = new CreationTool(new CERRelationshipSetFigure(CERRelationshipSet.TYPE_IDENTIFYING), attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
        
        // ISA
        t = new JToggleButton();
        cerLabels.configureToolBarButton(t, "edit.createISA");
        tool = new CreationTool(new CERISAFigure(), attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
        
        // AttributeKey
        t = new JToggleButton();
        cerLabels.configureToolBarButton(t, "edit.createAttribute");
        tool = new CreationTool(new CERAttributeFigure(), attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
        
        
        tb.addSeparator();
        
        // Connection
        t = new JToggleButton();
        cerLabels.configureToolBarButton(t, "edit.createConnection");
        tool = new ConnectionTool(new CERConnectionFigure(), attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
        
        // Many-to-Many Relationship
        t = new JToggleButton();
        cerLabels.configureToolBarButton(t, "edit.createRelationshipMM");
        tool = new CERRelationshipConnectionTool(new CERConnectionFigure(), attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
        
        // One-to-One Relationship
        t = new JToggleButton();
        cerLabels.configureToolBarButton(t, "edit.createRelationship11");
        tool = new CERRelationshipConnectionTool(new CERConnectionFigure(1), attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
        
        // Generalization
        t = new JToggleButton();
        cerLabels.configureToolBarButton(t, "edit.createGeneralization");
        tool = new CERGeneralizationConnectionTool(new CERConnectionFigure(), attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
        
        
        // AttributeKey link
        t = new JToggleButton();
        cerLabels.configureToolBarButton(t, "edit.createAttributeLink");
        tool = new CERAttributeConnectionTool(new CERConnectionFigure(), attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
        /*
        tb.addSeparator();
         
         t = new JToggleButton();
        drawLabels.configureToolBarButton(t, "edit.createText");
        tool = new TextTool(new TextFigure());
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
         
        t = new JToggleButton();
        drawLabels.configureToolBarButton(t, "edit.createTextArea");
        tool = new TextAreaTool(new TextAreaFigure());
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
         */
    }
    protected Drawing getDrawing() {
        return problemDiagram;
    }
    
    @Override
    public String getAppletInfo() {
        return NAME+"\nVersion "+VERSION
                +"\n\nCopyright 2006 \u00a9 Lucerne University of Applied Sciences and Arts"
                +"\nAlle Rechte vorbehalten."
                +"\n\nDiese Software basiert auf"
                +"\nJHotDraw \u00a9 1999-2008 by the original authors of JHotDraw and all its contributors."
                +"\nNanoXML \u00a9 2000-2002 Marc De Scheemaecker";
    }
    
    public void setData(String text) {
        problemDiagram.removeAllChildren();
        if (text != null && text.length() > 0) {
            StringReader in = new StringReader(text);
            try {
                undoManager.discardAllEdits();
                NanoXMLDOMInput domi = new NanoXMLDOMInput(new CERFactory(), in);
                domi.openElement("ConceptModelerCER");
                
                problemDiagram = (ConceptualERDiagram) domi.readObject(0);
                drawingView.setDrawing(problemDiagram);
                problemDiagram.addUndoableEditListener(undoManager);
            } catch (Throwable e) {
                problemDiagram.removeAllChildren();
                TextFigure tf = new TextFigure();
                tf.setText(e.getMessage());
                tf.setBounds(new Point2D.Double(10,10), new Point2D.Double(100,100));
                problemDiagram.add(tf);
                e.printStackTrace();
            } finally {
                if (in != null) in.close();
            }
        }
    }
    public String getData() {
        CharArrayWriter out = new CharArrayWriter();
        try {
            NanoXMLDOMOutput domo = new NanoXMLDOMOutput(new CERFactory());
            domo.openElement("ConceptModelerCER");
            domo.writeObject(problemDiagram);
            domo.closeElement();
            domo.save(out);
        } catch (IOException e) {
            TextFigure tf = new TextFigure();
            tf.setText(e.getMessage());
            tf.setBounds(new Point2D.Double(10,10), new Point2D.Double(100,100));
            problemDiagram.add(tf);
            e.printStackTrace();
        } finally {
            if (out != null) out.close();
        }
        return out.toString();
    }
    private Collection<Action> getDrawingActions() {
        if (drawingActions == null) {
            drawingActions = new LinkedList<Action>();
            drawingActions.add(undoManager.getUndoAction());
            drawingActions.add(undoManager.getRedoAction());
            drawingActions.add(null); // separator
            drawingActions.add(new SelectAllAction());
        }
        return drawingActions;
    }
    private Collection<Action> getSelectionActions() {
        if (selectionActions == null) {
            selectionActions = new LinkedList<Action>();
            selectionActions.add(new DuplicateAction());
            selectionActions.add(new DeleteAction());
        }
        return selectionActions;
    }
    
    @Override
    public String[][] getParameterInfo() {
        return new String[][] {
            { "locale", "String", "The locale of the apple, e.g. \"en\" for English." },
            { "data", "String", "The data to be displayed by this applet." },
            { "datafile", "URL", "An URL to the data to be displayed by this applet." },
            { "stylesheet", "URL", "An URL to the CSS stylesheet for this applet." },
        };
    }
    
    public static void main(String[] args) {
        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
                JFrame f = new JFrame(NAME+" "+VERSION);
                f.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
                f.setSize(500,300);
                CERApplet applet = new CERApplet();
                f.getContentPane().add(applet);
                applet.init();
                f.setVisible(true);
                applet.start();
            }
        });
    }
    
    public void toolStarted(ToolEvent event) {
    }
    
    protected JPanel getEditorPanel() {
        return editorPanel;
    }
    
    /** This method is called from within the init() method to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code ">//GEN-BEGIN:initComponents
    private void initComponents() {
        toolGroup = new javax.swing.ButtonGroup();
        editorPanel = new javax.swing.JPanel();
        scrollPane = new javax.swing.JScrollPane();
        drawingView = new org.jhotdraw.draw.DefaultDrawingView();
        toolBar = new javax.swing.JToolBar();

        editorPanel.setLayout(new java.awt.BorderLayout());

        scrollPane.setViewportView(drawingView);

        editorPanel.add(scrollPane, java.awt.BorderLayout.CENTER);

        toolBar.setFloatable(false);
        editorPanel.add(toolBar, java.awt.BorderLayout.SOUTH);

        getContentPane().add(editorPanel, java.awt.BorderLayout.CENTER);

    }// </editor-fold>//GEN-END:initComponents
    
    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private org.jhotdraw.draw.DefaultDrawingView drawingView;
    private javax.swing.JPanel editorPanel;
    private javax.swing.JScrollPane scrollPane;
    private javax.swing.JToolBar toolBar;
    private javax.swing.ButtonGroup toolGroup;
    // End of variables declaration//GEN-END:variables
    
}
